/*******************************************************************************
*  Sound.c
*
*  :   
*
*       Copyright (c) 2008 Nick Shl
*           All rights reserved.
*
*
*  :
*
*  May 07, 2008  Nick_Shl   
*
*/// ***************************************************************************

// *****************************************************************************
// ***       ***************************************************
// *****************************************************************************
#include <mega128.h>
#include <delay.h>
#include <stdio.h>

// *****************************************************************************
// ***       ********************************************
// *****************************************************************************
#include "Def.h"
#include "Variables.h"
#include "Sound.h"

// *****************************************************************************
// ***       ***********************************************
// *****************************************************************************
static flash unsigned int * SoundTable;
static unsigned int SoundTableSize;
static unsigned int SoundTablePosition;

// *****************************************************************************
// ***    Buzzer'   Freq   **********************************
// *****************************************************************************
void Tone(unsigned int Freq)
{
    //    :
    // Clock source: System Clock
    // Mode: CTC top = OCR2
    // OC2 output: Toggle on compare match
    //       ,  :
    //   64     ~750
    //   256   > ~1500   
    //   4   ,   AVR_Clock_Freq/x   word
    //     ,    
    //    -       
    if((Settings.SoundFlag == ON) && (Freq > 1000))
    {
        //    
        OCR2 = (((AVR_Clock_Freq/64) >> 2) / (Freq >> 2)) >> 1;
        //  (Clock divisor: 64):
        TCCR2 = 0b00011011;
    }
    else if((Settings.SoundFlag == ON) && Freq)
    {
        //    
        OCR2 = ((AVR_Clock_Freq/256) / Freq) >> 1;
        //  (Clock divisor: 256):
        TCCR2 = 0b00011100;
    }
    else
    {
        //  
        TCCR2 = 0b00000000;
        //    , ..       
        PORTB.7 = 0;
    }
}

// *****************************************************************************
// ***    Buzzer'  Del ,   Freq   ***************
// *****************************************************************************
void Beep(unsigned int Freq, unsigned int Del, unsigned char PauseAfterPlay)
{
    //    - 
    if(Settings.SoundFlag == OFF) return;

    Tone(Freq);
    delay_ms(Del); // 
    Tone(0);

    //    -       
    if(PauseAfterPlay) delay_ms(Del);
}

// *****************************************************************************
// ***          *******************************
// *****************************************************************************
void PlaySound(flash unsigned int * Melody, unsigned int N)
{
    //    -       .
    //    0x***#,  *** , # -   40 
    // 

    //      - 
    if(Melody == NULL) return;

    //    ,    - 
    if(IsSoundPlayed() == TRUE) StopSound();

    //   
    SoundTablePosition = 0;
    //   
    SoundTableSize = N;
    //     -   , ..
    //      
    SoundTable = Melody;
}

// *****************************************************************************
// ***           ***********************
// *****************************************************************************
void StopSound(void)
{
    //     -   , .. 
    //     
    SoundTable = (flash unsigned int *)NULL;
    //   
    SoundTableSize = 0;
    //   
    SoundTablePosition = 0;
    //   
    Tone(0);
}

// *****************************************************************************
// ***      -  TRUE,  - FALSE   *********
// *****************************************************************************
inline char IsSoundPlayed(void)
{
    //       -  
    if(SoundTable != NULL) return(TRUE);
    //  -  
    return(FALSE);
}

// *****************************************************************************
// ***          *******************************
// *****************************************************************************
void PlaySound_Task(unsigned char MissedInterruptsCount)
{
    static char Delay;

    //      - 
    if(SoundTable == NULL) return;
    //      - 
    if(SoundTableSize == 0) return;

    //   :     TASK_TICK_TIME  + 
    Delay -= (MissedInterruptsCount + 1);
    //      -    .
    if(Delay > 0) return;

    //    18  -  .  - .
    if((SoundTable[SoundTablePosition] >> 4) > 0x12) Tone(SoundTable[SoundTablePosition] >> 4);
    else                                             Tone(0);

    //     TASK_TICK_TIME*3  
    // FIX ME:    TASK_TICK_TIME*3
    Delay = ((SoundTable[SoundTablePosition] & 0x0F) + 1) * 3;

    //    
    SoundTablePosition++;
    //      - 
    if(SoundTablePosition > SoundTableSize) StopSound();
}

// *****************************************************************************
// ***    Buzzer'      **************************
// *****************************************************************************
void trim_sound(void)
{
    if(trim_sound_en && (Settings.SoundFlag == ON))
    {
        trim_sound_en=0;
        Tone(1000);     //    1kHz
        delay_ms(30);   // 
        Tone(0);        //  
        delay_ms(30);   // 
    }
}
