/*******************************************************************************
*  System.c
*
*  :  
*
*       Copyright (c) 2009 Nick Shl
*           All rights reserved.
*
*
*  :
*
*  Mar 25, 2009  Nick_Shl   
*
*/// ***************************************************************************

#include <mega128.h>
#include <delay.h>
#include <stdio.h>
#include <string.h>

#include "Def.h"
#include "Variables.h"
#include "Graphic.h"
#include "Sound.h"
#include "Tasks.h"
#include "UI_Engine.h"
#include "Coder.h"
#include "CRC.h"
#include "System.h"

// *****************************************************************************
// ***        ( )   **********
// *****************************************************************************
static flash char * DefModeNames[MAX_MODES] =
{
    "Normal",
    "Pilotage",
    "Landing"
};

// *****************************************************************************
// ***     -       ***********
// *****************************************************************************
static eeprom unsigned char EEPROM_TRASH @(0x0000);

// *****************************************************************************
// ***      CRC,    EEPROM   ***************
// *****************************************************************************
static eeprom unsigned short EEPROM_CRC @(0x1000 - sizeof(unsigned short));

// *****************************************************************************
// ***        ********************************************
// *****************************************************************************
//    EEPROM     CRC
//    EEPROM(0) +   
#define EEPROM_CRC_START ((unsigned char eeprom *)(0x0000 + sizeof(EEPROM_TRASH)))
//   EEPROM     CRC
//   EEPROM -    -    CRC
#define EEPROM_CRC_SIZE (0x1000 - sizeof(EEPROM_TRASH) - sizeof(EEPROM_CRC))

// *****************************************************************************
// ***    CRC EEPROM      ************************************
// *****************************************************************************
static inline void Recalc_EEPROM_CRC(void)
{
    //   EEPROM CRC     EEPROM
    EEPROM_CRC = Crc16_eeprom(EEPROM_CRC_START, EEPROM_CRC_SIZE);
}

// *****************************************************************************
// ***    CRC EEPROM   **************************************************
// *****************************************************************************
inline unsigned short Get_EEPROM_CRC(void)
{
    //  EEPROM CRC
    return(Crc16_eeprom(EEPROM_CRC_START, EEPROM_CRC_SIZE));
}

// *****************************************************************************
// ***     CRC EEPROM   ************************************
// *****************************************************************************
inline unsigned char Is_EEPROM_CRC_Correct(void)
{
    //  EEPROM CRC    
    //   -  
    if(EEPROM_CRC == Crc16_eeprom(EEPROM_CRC_START, EEPROM_CRC_SIZE)) return(TRUE);
    //  - 
    else return(FALSE);
}

// *****************************************************************************
// ***      EEPROM       ********
// *****************************************************************************
void TX_Init(void)
{
    //   
    unsigned char i;
    //  
    unsigned char eeprom * p_from;
    //  
    unsigned char        * p_to;

    //     
    #asm("cli")

    //       EEPROM
    p_from = (unsigned char eeprom *)&EEPROM_SETTINGS;
    //       SRAM
    p_to   = (unsigned char *)&Settings;

    //     EEPROM
    for(i=0; i < sizeof(TRANSMITTER_SETTINGS); i++)
    {
        *(p_to++) = *(p_from++);
    }

    // /     
    gfx_BackLight(Settings.BacklightFlag);

    //  
    MODEL_Init(Settings.ModelNum);

    //    
    #asm("sei")
}

// *****************************************************************************
// ***       EEPROM   ****************************
// *****************************************************************************
void TX_SaveSettings(void)
{
    //   
    unsigned char i;
    //  
    unsigned char        * p_from;
    //  
    unsigned char eeprom * p_to;

#ifdef DEBUG
    printf("TX_SaveSettings();\r");
    delay_ms(1);
#endif

    //     
    #asm("cli")

    //       SRAM
    p_from = (unsigned char *)&Settings;
    //       EEPROM
    p_to   = (unsigned char eeprom *)&EEPROM_SETTINGS;

    //     EEPROM
    for(i=0; i < sizeof(TRANSMITTER_SETTINGS); i++)
    {
        *(p_to++) = *(p_from++);
    }

    //   CRC EEPROM   
    Recalc_EEPROM_CRC();

    //    
    #asm("sei")
}

// *****************************************************************************
// ***       *************************************************
// *****************************************************************************
void TX_SetModulation(char Modulation)
{
    //         
    if(Modulation == MODUL_PPM)  TCCR1A=0x82;
    if(Modulation == MODUL_IPPM) TCCR1A=0xC2;
}

// *****************************************************************************
// ***      EEPROM   *******************************************
// *****************************************************************************
void MODEL_Init(char ModelNum)
{
    //   
    int i;
    //  
    unsigned char eeprom * p_from;
    //  
    unsigned char        * p_to;

#ifdef DEBUG
    printf("MODEL_Init(%d);\r", ModelNum);
    delay_ms(1);
#endif

    //     
    #asm("cli")

    //       EERPOM
    p_from = (unsigned char eeprom *)&EEPROM_MODEL[ModelNum];
    //       SRAM
    p_to   = (unsigned char *)&CurModel;

    //     EEPROM  SRAM
    for(i=0; i < sizeof(MODEL_SETTINGS); i++)
    {
        *(p_to++) = *(p_from++);
    }

    //     
    TX_SetModulation(CurModel.modulation);

    //  
    Timer_Init();

    //    
    #asm("sei")
}

// *****************************************************************************
// ***        EEPROM   ************************
// *****************************************************************************
void MODEL_Save_Trimmers(void)
{
    //   
    int i, j;

#ifdef DEBUG
    printf("MODEL_Save_Trimmers();\r");
    delay_ms(1);
#endif

    for(i=0; i < MAX_MODES; i++)
    {
        for(j=0; j < MAX_TRIMMERS; j++)
        {
            EEPROM_MODEL[Settings.ModelNum].Mode[i].trimmers[j] = CurModel.Mode[i].trimmers[j];
        }
    }

    //   CRC EEPROM   
    Recalc_EEPROM_CRC();
}


// *****************************************************************************
// ***      EEPROM   ******************************************
// *****************************************************************************
void MODEL_Save(char To)
{
    //   
    int i;
    //  
    unsigned char        * p_from;
    //  
    unsigned char eeprom * p_to;

#ifdef DEBUG
    printf("MODEL_Save(%d);\r", To);
    delay_ms(1);
#endif

    //      
    #asm("cli")

    //        SRAM
    p_from = (unsigned char *)&CurModel;
    //       EEPROM
    p_to   = (unsigned char eeprom *)&EEPROM_MODEL[To];

    //     SRAM  EEPROM
    for(i=0; i < sizeof(MODEL_SETTINGS); i++)
    {
        *(p_to++) = *(p_from++);
    }

    //   CRC EEPROM   
    Recalc_EEPROM_CRC();

    //    
    #asm("sei")
}

// *****************************************************************************
// ***       **************************************************
// *****************************************************************************
void MODEL_Copy(char From, char To)
{
    //   
    int i;
    //    
    unsigned char eeprom * p_from, * p_to;

#ifdef DEBUG
    printf("MODEL_Copy(%d, %d);\r", From, To);
    delay_ms(1);
#endif

    //      
    #asm("cli")

    //       EEPROM
    p_from = (unsigned char eeprom *)&EEPROM_MODEL[From];
    //       EEPROM
    p_to   = (unsigned char eeprom *)&EEPROM_MODEL[To];

    //   
    for(i=0; i < sizeof(MODEL_SETTINGS); i++)
    {
        *(p_to++) = *(p_from++);
    }

    //   CRC EEPROM   
    Recalc_EEPROM_CRC();

    //    
    #asm("sei")
}

// *****************************************************************************
// ***        ****************************************
// *****************************************************************************
void MODE_Copy(char From, char To)
{
    //   
    int i;
    //    ( 12  + -)
    char Name[MODE_NAME_LEN];
    //    
    unsigned char eeprom * p_from, * p_to;

#ifdef DEBUG
    printf("MODE_copy(%d, %d);\r", From, To);
    delay_ms(1);
#endif

    //       
    #asm("cli")

    //      
    for(i=0; i < MODE_NAME_LEN; i++)
    {
        Name[i] = EEPROM_MODEL[Settings.ModelNum].Mode[To].name[i];
    }

    //        EEPROM
    p_from = (unsigned char eeprom *)&EEPROM_MODEL[Settings.ModelNum].Mode[From];
    //        EEPROM
    p_to   = (unsigned char eeprom *)&EEPROM_MODEL[Settings.ModelNum].Mode[To];

    //    
    for(i=0; i < sizeof(DEF_MODE); i++)
    {
        *(p_to++) = *(p_from++);
    }

    //      
    for(i=0; i < MODE_NAME_LEN; i++)
    {
        EEPROM_MODEL[Settings.ModelNum].Mode[To].name[i] = Name[i];
    }

    //   CRC EEPROM   
    Recalc_EEPROM_CRC();

    //    
    #asm("sei")
}

// *****************************************************************************
// ***          *********************************
// *****************************************************************************
void MODEL_Reset(char ModelNum)
{
    //   
    unsigned char i;
    unsigned char len;
    unsigned char MOD, CH, CTRL;

#ifdef DEBUG
    printf("MODEL_Reset(%d);\r", ModelNum);
    delay_ms(1);
#endif

    //      
    #asm("cli")

        for(MOD=0; MOD < MAX_MODES; MOD++)
        {
            for(CTRL=0; CTRL < MAX_CONTROLS; CTRL++)
                {
                    //  1 -  
                    EEPROM_MODEL[ModelNum].Mode[MOD].Control[CTRL].reverse = 1;

                    //  
                    EEPROM_MODEL[ModelNum].Mode[MOD].Control[CTRL].maxRates = 100;
                    EEPROM_MODEL[ModelNum].Mode[MOD].Control[CTRL].minRates = 100;  // EPA val in 2% unit

                    //   
                    EEPROM_MODEL[ModelNum].Mode[MOD].Control[CTRL].maxDRates = 100; // EPA val in 2% unit
                    EEPROM_MODEL[ModelNum].Mode[MOD].Control[CTRL].minDRates = 100;

                    //  
                    for(i=0; i < CURVE_NODES; i++)
                    {
                        EEPROM_MODEL[ModelNum].Mode[MOD].Control[CTRL].nodes[i] = (200 * (int)i) / (CURVE_NODES - 1) - 100;
                    }

                    //   ( )
                    EEPROM_MODEL[ModelNum].Mode[MOD].Control[CTRL].from = CTRL;
                }

            //  
            for(CH=0; CH < MAX_CHANNELS; CH++)
                for(CTRL=0; CTRL < MAX_CONTROLS; CTRL++)
                    EEPROM_MODEL[ModelNum].Mode[MOD].Chanels[CH][CTRL] = 0;

            //    
            for(CH=0; CH < MAX_CHANNELS; CH++) EEPROM_MODEL[ModelNum].Mode[MOD].Chanels[CH][CH] = 100;

            //  
            for(i=0; i < 4; i++) EEPROM_MODEL[ModelNum].Mode[MOD].trimmers[i] = 0;

            //    
            for(i=0; i < MODE_NAME_LEN; i++)
            {
                EEPROM_MODEL[ModelNum].Mode[MOD].name[i] = DefModeNames[MOD][i];
                if(DefModeNames[MOD][i] == 0x00) break;
            }
            //       
            for(; i < MODE_NAME_LEN; i++) EEPROM_MODEL[ModelNum].Mode[MOD].name[i] = 0x00;
        }

        //  
        sprintf(tmpBuf, "Model %d", ModelNum + 1);
        len = (unsigned char)strlen(tmpBuf);
        if(len > NumberOf(EEPROM_MODEL[ModelNum].name) - 1) len = NumberOf(EEPROM_MODEL[ModelNum].name) - 1;
        for(i=0; i < len; i++) EEPROM_MODEL[ModelNum].name[i] = tmpBuf[i];
        for(; i < NumberOf(EEPROM_MODEL[ModelNum].name); i++) EEPROM_MODEL[ModelNum].name[i] = 0;

        //   
        EEPROM_MODEL[ModelNum].modulation = MODUL_PPM;

        //   
        EEPROM_MODEL[ModelNum].type = TYPE_PLANE;

        //   
        EEPROM_MODEL[ModelNum].timer = 0;
        EEPROM_MODEL[ModelNum].timer_mode = 0;

        //   
        EEPROM_MODEL[ModelNum].num_ch = 6;

        //   
        EEPROM_MODEL[ModelNum].size = sizeof(EEPROM_MODEL[ModelNum]);

        //   CRC EEPROM   
        Recalc_EEPROM_CRC();

    //    
    #asm("sei")
}

// *****************************************************************************
// ***      **********************************************************
// *****************************************************************************
void Calibration(void)
{
    unsigned char i, x, y;

#ifdef DEBUG
    printf("Calibration();\r");
    delay_ms(1);
#endif

    //     
    #asm("cli")

    for(i=0; i < MAX_ADC; i++)
    {
        AD[i]    = read_adc(i);
        ADmin[i] = AD[i];
        ADmax[i] = AD[i];
    }

    gfx_ClearBuf();
    gfx_SetXY((WidthS - 11*Font_8x12.W) >> 1, (HeightS - 1*Font_8x12.H) >> 1);
    gfx_PutFlashStr("Calibrating", 0xFF, &Font_8x12);
    gfx_Refresh();
    delay_ms(1000);

    while(!HB_ENTER)
    {
        gfx_ClearBuf();

        for (i=0; i < MAX_ADC; i++)
        {
            AD[i] = read_adc(i);
            if(ADmin[i] > AD[i]) ADmin[i] = AD[i];
            if(ADmax[i] < AD[i]) ADmax[i] = AD[i];
        }

        gfx_Rectangle(0, 4, 55, 59, 1);
        gfx_Rectangle(1, 5, 54, 58, 1);

        gfx_Rectangle(72, 4, 127, 59, 1);
        gfx_Rectangle(73, 5, 126, 58, 1);

        x = (unsigned char)(     (unsigned int)(AD[ADC_RUD] - ADmin[ADC_RUD]) * 52 / (ADmax[ADC_RUD] - ADmin[ADC_RUD] + 1) + 2);
        y = (unsigned char)(52 - (unsigned int)(AD[ADC_THR] - ADmin[ADC_THR]) * 52 / (ADmax[ADC_THR] - ADmin[ADC_THR] + 1) + 5);

        gfx_PutPixel(x, y - 2, 1);
        gfx_PutPixel(x, y + 2, 1);
        gfx_Line(x - 1, y - 1, x + 1, y - 1, 1);
        gfx_Line(x - 2, y,     x + 2, y,     1);
        gfx_Line(x - 1, y + 1, x + 1, y + 1, 1);

        x = (unsigned char)(     (unsigned int)(AD[ADC_AIL] - ADmin[ADC_AIL]) * 52 / (ADmax[ADC_AIL] - ADmin[ADC_AIL] + 1) + 74);
        y = (unsigned char)(52 - (unsigned int)(AD[ADC_ELE] - ADmin[ADC_ELE]) * 52 / (ADmax[ADC_ELE] - ADmin[ADC_ELE] + 1) +  5);

        gfx_PutPixel(x, y - 2, 1);
        gfx_PutPixel(x, y + 2, 1);
        gfx_Line(x - 1, y - 1, x + 1, y - 1, 1);
        gfx_Line(x - 2, y    , x + 2, y    , 1);
        gfx_Line(x - 1, y + 1, x + 1, y + 1, 1);

        y = 1 + (unsigned char)((unsigned int)(AD[ADC_AUX1] - ADmin[ADC_AUX1]) * 61 / (ADmax[ADC_AUX1] - ADmin[ADC_AUX1] + 1));

        gfx_Rectangle(59, 0, 61, 63, 1);
        gfx_FillRect(58, y, 62, y + 1, 1);

        y = 1 + (unsigned char)((unsigned int)(AD[ADC_AUX2] - ADmin[ADC_AUX2]) * 61 / (ADmax[ADC_AUX2] - ADmin[ADC_AUX2] + 1));

        gfx_Rectangle(66, 0, 68, 63, 1);
        gfx_FillRect(65, y, 69, y + 1, 1);

        gfx_Refresh();
    }

    //  
    gfx_ClearBuf();
    //  
    MsgBoxF("Center all\nsticks and\npress Enter", NULL, "Calibration", NULL);
    //           //  \
    while(HB_ENTER);                 //   |      ,  
    //         //   |      .
    while(!HB_ENTER);                //  /

    for(i=0; i < MAX_ADC; i++)
    {
        //       
        Settings.ADmid[i] = read_adc(i);
        EEPROM_SETTINGS.ADmid[i] = Settings.ADmid[i];
        //     10,     .
        //        10.
        //       
        //       
        Settings.Kmax[i] = (TimerClockPerSec(0.0005) * 10) / (ADmax[i] - Settings.ADmid[i]); // \
        EEPROM_SETTINGS.Kmax[i] = Settings.Kmax[i];                                          //  |
        //                              //  | FIX ME:    
        //                              //  |         ?
        Settings.Kmin[i] = (TimerClockPerSec(0.0005) * 10) / (Settings.ADmid[i] - ADmin[i]); //  |
        EEPROM_SETTINGS.Kmin[i] = Settings.Kmin[i];                                          // /
    }

    //   CRC EEPROM   
    Recalc_EEPROM_CRC();

    //    
    #asm("sei")
}

// *****************************************************************************
// ***           *******************************
// *****************************************************************************
void TX_Reset(void)
{
    //   
    int i;
    int StartPos = (HeightS - Font_8x12.H)/2 - Font_8x12.H;

#ifdef DEBUG
    printf("TX_Reset();\r");
    delay_ms(1);
#endif

    //         
    #asm("cli")
        Settings.SoundFlag = ON;

        //  
        gfx_ClearBuf();

        gfx_SetXY((WidthS - Font_8x12.W*10)/2, StartPos);
        gfx_PutFlashStr("Please", 0xFF, &Font_8x12);
        gfx_SetXY((WidthS - Font_8x12.W*5)/2, StartPos + Font_8x12.H);
        gfx_PutFlashStr("wait...", 0xFF, &Font_8x12);
        gfx_Rectangle(0, StartPos + Font_8x12.H*2, WidthS - 1, StartPos + Font_8x12.H*3, 0xFF);
        gfx_Refresh();

        for(i=0; i < MAX_MODELS; i++)
        {
            Settings.ModelNum = i;
            MODEL_Reset(Settings.ModelNum);
            gfx_FillRect(0, StartPos + Font_8x12.H*2, (WidthS * (i + 1) - 1) / MAX_MODELS, StartPos + Font_8x12.H*3, 0xFF);
            gfx_Refresh();
        }

        delay_ms(500);

        EEPROM_SETTINGS.size = sizeof(EEPROM_SETTINGS);
        EEPROM_SETTINGS.ModelNum = 0;
        EEPROM_SETTINGS.SoundFlag = ON;
        EEPROM_SETTINGS.BacklightFlag = OFF;
        Calibration();

        EEPROM_SETTINGS.FirstON = 1;

        //   CRC EEPROM   
        Recalc_EEPROM_CRC();

    //    
    #asm("sei")
}
