/*******************************************************************************
*  Coder.c
*
*  :  
*
*       Copyright (c) 2007-2008 Nick Shl, focus
*           All rights reserved.
*
*
*  :
*
*  Apr 20, 2009  Nick_Shl  
*  Apr 10, 2008  Nick_Shl  , .
*  *** **, ****  focus      
*
*  Chip type           : ATmega128
*  Program type        : Application
*  Clock frequency     : 16,000000 MHz
*  Memory model        : Small
*  External SRAM size  : 0
*  Data Stack size     : 1024
*/// ***************************************************************************

// *****************************************************************************
// ***       ***************************************************
// *****************************************************************************
#include <mega128.h>
#include <delay.h>
#include <stdio.h>

// *****************************************************************************
// ***     COM  -  CodeVisionAVR   *******************
// *****************************************************************************
#define RXB8 1
#define TXB8 0
#define UPE 2
#define OVR 3
#define FE 4
#define UDRE 5
#define RXC 7

#define FRAMING_ERROR (1 << FE)
#define PARITY_ERROR (1 << UPE)
#define DATA_OVERRUN (1 << OVR)
#define DATA_REGISTER_EMPTY (1 << UDRE)
#define RX_COMPLETE (1 << RXC)

// USART0 Receiver buffer
#define RX_BUFFER_SIZE0 32
char rx_buffer0[RX_BUFFER_SIZE0];

#if RX_BUFFER_SIZE0 < 256
    unsigned char rx_wr_index0, rx_rd_index0, rx_counter0;
#else
    unsigned int rx_wr_index0, rx_rd_index0, rx_counter0;
#endif

// This flag is set on USART0 Receiver buffer overflow
bit rx_buffer_overflow0;

// USART0 Receiver interrupt service routine
interrupt [USART0_RXC] void usart0_rx_isr(void)
{
    char status, data;
    status = UCSR0A;
    data = UDR0;
    if((status & (FRAMING_ERROR | PARITY_ERROR | DATA_OVERRUN)) == 0)
    {
        rx_buffer0[rx_wr_index0] = data;
        if(++rx_wr_index0 == RX_BUFFER_SIZE0) rx_wr_index0 = 0;
        if(++rx_counter0 == RX_BUFFER_SIZE0)
        {
            rx_counter0 = 0;
            rx_buffer_overflow0 = 1;
        };
    };
}

#ifndef _DEBUG_TERMINAL_IO_
// Get a character from the USART0 Receiver buffer
#define _ALTERNATE_GETCHAR_
#pragma used+
char getchar(void)
{
    char data;
    while(rx_counter0 == 0);
    data = rx_buffer0[rx_rd_index0];
    if (++rx_rd_index0 == RX_BUFFER_SIZE0) rx_rd_index0 = 0;
    #asm("cli")
    --rx_counter0;
    #asm("sei")
    return data;
}
#pragma used-
#endif

// USART0 Transmitter buffer
#define TX_BUFFER_SIZE0 32
char tx_buffer0[TX_BUFFER_SIZE0];

#if TX_BUFFER_SIZE0 < 256
    unsigned char tx_wr_index0, tx_rd_index0, tx_counter0;
#else
    unsigned int tx_wr_index0, tx_rd_index0, tx_counter0;
#endif

// USART0 Transmitter interrupt service routine
interrupt [USART0_TXC] void usart0_tx_isr(void)
{
    if(tx_counter0)
    {
        --tx_counter0;
        UDR0 = tx_buffer0[tx_rd_index0];
        if(++tx_rd_index0 == TX_BUFFER_SIZE0) tx_rd_index0 = 0;
    };
}

#ifndef _DEBUG_TERMINAL_IO_
// Write a character to the USART0 Transmitter buffer
#define _ALTERNATE_PUTCHAR_
#pragma used+
void putchar(char c)
{
    while(tx_counter0 == TX_BUFFER_SIZE0);
    #asm("cli")
    if(tx_counter0 || ((UCSR0A & DATA_REGISTER_EMPTY) == 0))
    {
        tx_buffer0[tx_wr_index0] = c;
        if(++tx_wr_index0 == TX_BUFFER_SIZE0) tx_wr_index0 = 0;
        ++tx_counter0;
    }
    else UDR0 = c;
    #asm("sei")
}
#pragma used-
#endif

// *****************************************************************************
// ***       COM    **************************************
// *****************************************************************************

// *****************************************************************************
// ***       ********************************************
// *****************************************************************************
#include "Def.h"
#include "Hardware.h"
#include "Variables.h"
#include "Graphic.h"
#include "Sound.h"
#include "Math.h"
#include "System.h"
#include "Crc.h"
#include "Tasks.h"
#include "UI_Engine.h"
#include "User_Interface.h"

// *****************************************************************************
// ***       ***********************************************
// *****************************************************************************
static unsigned long TickCount = 0;
flash unsigned int TxStartSound[] = {784<<4 + 2, 660<<4 + 1, 523<<4 + 1, 784<<4 + 2};

// *****************************************************************************
// ***       ***************************************************
// *****************************************************************************

// *****************************************************************************
// ***      *********************************************************
// *****************************************************************************

// *****************************************************************************
// ***            ************
// *****************************************************************************
inline unsigned long GetTickCount(void)
{
#ifdef DEBUG
    return(TickCount / TASK_TICK_TIME);
#else
    return(TickCount);
#endif
}

// *****************************************************************************
// ***            ***************
// *****************************************************************************
inline unsigned long GetRunningTime(void)
{
#ifdef DEBUG
    return(TickCount);
#else
    return(TickCount * TASK_TICK_TIME);
#endif
}

// *****************************************************************************
// ***                                                      ***
// ***           :)          ***
// *****************************************************************************
interrupt [TIM1_COMPA] void timer1_compa_isr(void)
{
    //   
    static unsigned char nb = 0;
    //     PPM 
    static unsigned int del = 0;

#ifdef DEBUG
    //  
    static unsigned char gauge = 0; //  
    static char dir = 1;            //  
#endif

    if(nb < CurModel.num_ch)
    {
        //    - 
        math_CalcChannel(nb);

        ICR1H = output[nb] >> 8;         //    (Hi)
        ICR1L = output[nb] & 0b11111111; //    (Lo)

        del += output[nb]; //        

        nb++; //   
    }
    else
    {
        del = TimerClockPerSec(0.020) - del; //  :  
                                             //  0.020 (20 ) -   
        ICR1H = del >> 8;                    //    (Hi)
        ICR1L = del & 0b11111111;            //    (Lo)

        del = 0; //     
        nb = 0;  //   

#ifdef DEBUG
        //     -    -.
        if(gauge == 0x00) dir = +1;
        if(gauge == 0xFF) dir = -1;
        GAUGE(gauge);
        gauge += dir;
#endif

        //    
        math_CalcControls();

        //      :
        // (    *  ) /    
        Aref = (123L * 1023L) / (long)read_adc(0x1E);
    }
}

// *****************************************************************************
// ***                                                      ***
// ***         ,   ***
// ***      ,       ..    ***
// ***         .     ***
// ***   FIX ME:                ***
// ***   timer1_compa_isr   :                                 ***
// ***   if(nb == 0) timer3_compa_isr();                                     ***
// *****************************************************************************
interrupt [TIM3_COMPA] void timer3_compa_isr(void)
{
    static unsigned char InterruptEnterFlag = 0;
    static unsigned char MissedIntCount = 0;
    unsigned char MissedInterruptsCount = 0;

    //      TASK_TICK_TIME   (DEBUG) 
    TickCount++;

#ifdef DEBUG
    if(TickCount % TASK_TICK_TIME) return;
#endif

    //        -   
    //    .
    if(InterruptEnterFlag)
    {
        MissedIntCount++;
        return;
    }

    //   ,       
    InterruptEnterFlag = 1;
    //       ,
    // ..        .
    MissedInterruptsCount = MissedIntCount;
    //    .
    MissedIntCount = 0;

    //       -    
    #asm("sei")

    //   
    Keyboard_Driver_Task(MissedInterruptsCount);

    //    
    Timer_Task(MissedInterruptsCount);

    //    
    PlaySound_Task(MissedInterruptsCount);

    //   
    Battery_Driver_Task(MissedInterruptsCount);

    //   
    Trimmers_Save_Task(MissedInterruptsCount);

    //    
//    gfx_AutoRefreshTask(MissedInterruptsCount);

    //  
    #asm("cli")
    //  C ,        
    InterruptEnterFlag = 0;

    return;
}

// *****************************************************************************
// ***       *****************************************************
// *****************************************************************************
void main(void)
{
    //     
    unsigned long LastRefreshTime = 0;
    //     
    unsigned char Kbd = 0;

// ***     **************************************************
    hwl_InitPorts(); // Ports initialization
    hwl_InitTimers(); // Timers/Counters initialization
    hwl_InitUSART0(BAUD_RATE); // USART0 initialization
    hwl_InitMisc(); // All other hardware initialization

    //   
#ifdef DEBUG
    printf("Started program...\r");
    printf("CRC FLASH:  0x%04X\r", Crc16_flash((unsigned char flash *)0, 0x1000 - sizeof(unsigned short)));
    printf("CRC EEPROM: 0x%04X ", Get_EEPROM_CRC());
    if(Is_EEPROM_CRC_Correct() == TRUE) printf("CORRECT\r");
    else printf("INCORRECT\r");
    if(sizeof(EEPROM_MODEL_SETTINGS) != sizeof(MODEL_SETTINGS)) printf("WARNING!!! ");
    printf("EEPROM_MODEL_SETTINGS(%d) == MODEL_SETTINGS(%d);\r", sizeof(EEPROM_MODEL_SETTINGS), sizeof(MODEL_SETTINGS));
    delay_ms(1);
#endif

    //   
    gfx_Init();
    //  
    gfx_ClearBuf();
    //    
    MsgBoxF("Starting...", NULL, "TRANSMITTER", NULL);

    //   ,       
    if((EEPROM_SETTINGS.FirstON != 1) || (HB_BACK && HB_ENTER)) TX_Reset();
    //      EEPROM  
    if(Is_EEPROM_CRC_Correct() == FALSE)
    {
        //    
        MsgBoxF("EEPROM CRC ERROR\nReset to factory\ndefaults ?", &Font_6x8, "ERROR", NULL);

        //   ,   
        while(HB_ENTER || HB_BACK);
        //   
        while(!HB_ENTER && !HB_BACK);
        // E   - 
        if(HB_ENTER) TX_Reset();
    }

    #asm("cli")
        //   (   EEPROM)
        TX_Init();
        //      
        TIMSK = 0x10;
        //     
        ETIMSK = 0x10;
        //     
        math_CalcControls();
    #asm("sei")

    //      
    delay_ms(50);

    //  
    PlaySound(TxStartSound, NumberOf(TxStartSound));

#ifdef DEBUG
//    StartTime = GetRunningTime();
//    printf("Execute time = %d ms\r", GetRunningTime() - LastRefreshTime);
#endif

    while(1)
    {
        //    
        Sleep();

        trim_sound();

        //        .     ,
        //      ,    .   ,  
        //       .
        do
        {
            //   100     
            if(LastRefreshTime + 100 <= GetRunningTime())
            {
                //    
                LastRefreshTime = GetRunningTime();
                //   
                MainScreen();
#ifdef DEBUG
//printf("MainScreen() execute time = %d ms (%d Ticks)\r", (GetTickCount() - LastRefreshTime) * TICK_TIME, GetTickCount() - LastRefreshTime);
#endif
            }
        }
        while(Kbd && AskButtons());

        //   
        Kbd = AskButtons();

        //       
        if(GetTimerSoundStatus() == On && Kbd)
        {
            //   
            TimerSoundOff();
            //    
            continue;
        }

        //       
        if(Kbd & B_LEFT) Timer_Init();
        //    
        if(Kbd & B_UP) Timer_Stop();
        //    
        if(Kbd & B_DOWN) Timer_Start();

        //       BACK
        //   BACK  /
        //     SRAM
        if(Kbd == B_BACK)
        {
            if(Settings.BacklightFlag == On) gfx_BackLight(Off);
            else                             gfx_BackLight(On);
        }

        //    
        if(Kbd == B_ENTER) MainMenu();
    };
}
